;**************************************************************************************************
;This program is used with the MCP41010 10K digitally controlled potentiometer.  This pot
;has a resolution of 8-bits.  The exercise demonstrates how a digitally
;controlled volume control works in a modern electronic devise.  The unit also
;demonstrates how Serial Peripheral Interface (SPI) protocol is used to communicate
;serially with a peripheral device such as this digital pot or other sensor.  There
;are three communications lines between the PIC and the peripheral device in the standard 
;SPI setup: a chip select (CS), a clock (CLK),and a data source (SI).
;
;This project uses the MCP41010 to control the volume of the tone produced in a speaker.
;The tone is generated by TMR0 interrupts (as developed in pervious chapters).  The main program 
;loop senses the status of two input switches (one for pot-up, the other for pot-down).
;Upon detecting a switch closure,the program calls the appropriate up/down routine which firsts
;turns on the tone and then increments or decrements the value being sent via SPI to set the
;digital potentiometer.  When the input switch is released, the interrupt that was generating the
;tone is disabled.  The byte value that is used to set the digital pot is also displayed on the 
;LCD display.
;*************************************************************************************************





	list      p=16F676       ; list directive to define processor
	#include <p16f676.inc>    ; processor specific variable definitions



	__CONFIG  _CP_OFF & _WDT_OFF & _BODEN & _PWRTE_ON & _INTRC_OSC_NOCLKOUT & _MCLRE_OFF & _CPD_OFF
;	__CONFIG  _CP_OFF & _WDT_OFF & _BODEN & _PWRTE_ON & _HS_OSC & _MCLRE_OFF & _CPD_OFF
; '__CONFIG' directive is used to embed configuration word within .asm file.
; The labels following the directive are located in the respective .inc file.
; See data sheet for additional information on configuration word settings.

;******************************************************************************
;Defines
;******************************************************************************

#define Bank0		0x00
#define	Bank1		0x80
#define	CS			0x03			;RC3 to CS
#define	SCK			0x04			;RC4 to SCK
#define	SI			0x05			;RC5 to SI
#define pot0		b'00010001'		;command to select Pot0
									;change this to 00010000 when using the MCP41010 maybe?
#define speaker		0x02			;RC2 to speaker
#define	counter 	0x08			;constant used in spi subroutine
#define TMR0_scale	.133			;TMR0 preload factor, this value gives 1000Hz toggle 
#define up			0x00			;PORTA,0 for the up switch
#define down		0x01			;PORTA,1 for the down switch

;*******************************************************************************
;LCD defines
;********************************************************************************
#define LCD			0x05			;pin RA5 connected to LCD data line
#define	LCD_BKSPC	0x08			;move cursor left
#define	LCD_RT		0x09			;move cursor right
#define	LCD_LF		0x0A			;move cursor down 1 line
#define LCD_CLS		0x0C			;clear LCD (allow 5 ms delay to complete)
#define	LCD_CR		0x0D			;move position 0 of next line
#define	LCD_BL_ON	0x11			;backlight on
#define	LCD_BL_OFF	0x12			;backlight off
#define LCD_OFF		0x15			;LCD off
#define LCD_ON1		0x16			;LCD on, cursor off, blink off
#define LCD_ON2		0x17			;LCD on, cursor off, blink on
#define LCD_ON3		0x18			;LCD on, cursor on, blink off
#define	LCD_ON4		0x19			;LCD on, cursor of, blink on
#define LCD_LINE0	0x80			;move to line 1, column 0, add position to 0x80
#define LCD_LINE1	0x94			;move to line 2, column 0, add position to 0x94
#define LCD_LINE2	0xA8			;move to line 3, column 0, add position to 0xA8
#define LCD_LINE3	0xBC			;move to line 4, column 0, add position to 0xBC
;end LCD defines
;******************************************************************************

;******************************************************************************
;General Purpose Registers (GPR's) 
;******************************************************************************

	cblock	0x20
	w_temp							;used as a temporary holder of w_reg during interrupts
	status_temp
	volume							;used as the volume byte sent to the digital pot
	data_to_send					;used in spi to receive data to send from the calling program line
	count							;used in delay routines
	tempa							;used in delay routines
	bitcounter						;used as bit counter in display routine
	byte_to_send					;used in desplay routines to receive data to send
	;********************************
	;used by Binary to BCD Converter
	countBCD
	temp
	h_byte
	l_byte
	r0
	r1
	r2
	;********************************
	endc


;******************************************************************************
;Reset Vector 
;******************************************************************************
	ORG     0x000         	; processor reset vector
	nop						; required by in circuit debugger  
	goto    Init            ; go to beginning of program

;******************************************************************************
;Interrupt Vector     
;******************************************************************************
	ORG	0x004
	goto interrupt_service
	return			; interrupt trap - returns without re-enabling
;******************************************************************************
;Initialization
;******************************************************************************
Init
	BANKSEL	Bank1
	call    0x3FF      ; retrieve factory calibration value
	movwf	OSCCAL
	BANKSEL	Bank0		;select bank0
	clrf	PORTA		;clear port bus
	movlw	b'00000111'	;comparator disconnect and lowest power
	movwf	CMCON		;
	movlw	b'00000000'	;globals, peripherals, ra2 int, clear INTF
	movwf	INTCON
	BANKSEL	Bank1		; BANK1
	movlw	b'00000001'	;TMR0 set-up:  pull-ups enabled,X,internal clk,X,
						;pre-scale tmr0, pre-scale 1:2 
	movwf	OPTION_REG	;put w reg into option register
	movlw	b'00001111'	;RA0, RA1, RA2, RA3 input, rest are output 
	movwf	TRISA		;program PORTA
	movwf	WPUA		;enable weak pull-up on RA0, RA1, RA2, RA3 - no pull-up
	movlw	b'00000000'	;all output
	movwf	TRISC		;program PORTC
	movlw	b'00000000'	;all digital pins
	movwf	ANSEL		
	BANKSEL	Bank0		;back to bank0
	movlw	.128
	movwf	volume		;set volume to the mid range for default value
;end pic initialization	
;*****************************************************************************
;Initialize the LCD display
;*****************************************************************************
	bsf		PORTA,LCD		;set to high for resting state
	call	wait1sec		;to allow LCD to inialize
	movlw	LCD_CLS
	call	LCDOutput
	call	delay5mS		;to allow LCD to complete clear
;end LCD initialization
;****************************************************************************


;*****************************************************************************
;main program
	movlw	TMR0_scale		;preload TMR0
	movwf	TMR0
	clrf	h_byte			;clear high byte of number to be displayed

	bsf		PORTC,speaker	;start with speaker off to keep idle current low
	bcf		PORTC,SCK		;start with pot clock bit low
	bsf		PORTC,CS		;start with pot chip select high (off)

	movlw	.4					;there are 4 control and characters for displaying Pot
	movwf	countBCD			;countBCD is used as a counter for using the table
	clrf	temp				;temp is used as a offset location within table
display_loop
	call	text_display		;gets a character to be displayed
	call	LCDOutput			;this call puts the character on the LCD
	incf	temp				;increment the offset to the next charater location
	decfsz	countBCD			;decrement the table counter, continue if not zero
	goto	display_loop

	movlw	LCD_LINE0+6			;moves cursor to 6th position on line 0
	call	LCDOutput
	movfw	volume				;puts the value of volume in low byte for display
	movwf	l_byte
	call	display_DEC
;end of LCD setup and display

main_loop
	btfss	PORTA,up			;check if up switch is pressed, skip if not
	goto	up_volume
	btfss	PORTA,down			;check if down switch is pressed, skip if not
	goto	down_volume
	goto	main_loop

up_volume
	movlw	TMR0_scale		;preload TMR0
	movwf	TMR0
	bcf		INTCON,T0IF		;clear TMR0 interrupt flag
	bsf		INTCON,T0IE		;enable TMR0
	bsf		INTCON,GIE		;enable global interrupts
repeat_up
	incf	volume,f		;increment the value in volume
	btfsc	STATUS,Z		;check if volume is incremented through zero
	decf	volume,f		;if it was, then set volume to zero

	movlw	LCD_LINE0+6		;send display to position 6 line 0
	call	LCDOutput	
	movfw	volume			;put value of volume in low byte for display
	movwf	l_byte
	call	display_DEC

	bcf		PORTC,CS		;select the digital pot
	movlw	pot0			;put command for pot0 in w_reg
	call	spi				;send command to pot
	movfw	volume			;get value of volume
	call	spi				;send it to the pot

	bsf		PORTC,CS		;de-slect the digital pot to allow it to reset resistance

	btfss	PORTA,up		;check if up switch is still pressed, if it is do it again
	goto	repeat_up

	bcf		INTCON,GIE		;if all done, disable tone
	bcf		INTCON,T0IE
	bcf		INTCON,T0IF
	goto	main_loop


down_volume
	movlw	TMR0_scale		;preload TMR0
	movwf	TMR0
	bcf		INTCON,T0IF		;clear TMR0 interrupt flag
	bsf		INTCON,T0IE		;enable TMR0
	bsf		INTCON,GIE
repeat_down
	movf	volume,f		
	btfss	STATUS,Z		;check if volume is zero
	decf	volume,f		;if it is not, then decrement volume

	movlw	LCD_LINE0+6
	call	LCDOutput
	movfw	volume
	movwf	l_byte
	call	display_DEC

	bcf		PORTC,CS
	movlw	pot0
	call	spi
	movfw	volume
	call	spi
	bsf		PORTC,CS

	btfss	PORTA,down
	goto	repeat_down

	bcf		INTCON,GIE
	bcf		INTCON,T0IE
	bcf		INTCON,T0IF
	goto	main_loop

interrupt_service
	movwf	w_temp				;copy contents of w_reg into a temp register
	swapf	STATUS,w			;swap the nibbles of STATUS and place into
								;the w_register, these nibbles will be swapped
								;back when the STATUS register is recovered
								;at the end of the interrupt service routine
	BANKSEL	Bank0				;forces a return to Bank 0 regardless of bank 
								;when interrupt occurred
	movwf	status_temp			;put the swapped old STATUS reg value in a temp
								;location, now with w and STATUS registers praotected
								;proceed with the interrupt service

	bcf		INTCON,T0IE			;disable TMR0 interrupt
	bcf		INTCON,T0IF			;clear TMR0 interrupt flag
	movlw	TMR0_scale			;reload TMR0
	movwf	TMR0
	movlw	b'00000100'	
	xorwf	PORTC,f				;toggle the speaker pin to create the tone
	
	swapf	status_temp,w		;swap the nibbles in status_temp and put result in w_reg
	movwf	STATUS				;STATUS now returned to pre-interrupt value
	swapf	w_temp,f			;take the old value of w_reg and swap nibbles
	swapf	w_temp,w			;swap nibbles again and place into w_reg, w_reg
								;now returned to pre-interrupt value
	bsf		INTCON,T0IE			;enable TMR0 interrupt
	retfie						;return from interrupt enables GIE
	


spi
	movwf	data_to_send		;put contents of w reg into working variable
	movlw	.8					;reset bit counter
	movwf	bitcounter
trans_loop
	bcf		PORTC,SI			;assume 0 bit
	rlf		data_to_send,f		;rotate command left into carry
	btfsc	STATUS, C			;if carry is high, set bit high/else skip
	bsf		PORTC,SI
	bsf		PORTC,SCK			;clock in the bit
	bcf		PORTC,SCK
	decfsz	bitcounter,f		;check if 8 bits sent, if not, go back
	goto	trans_loop
	return

		
;*****************************************************************************
; 
;use this routine to call the static text that is contained in table.
;the programming technique avoids a problem with using the program counter
;to jump across memory pages.  The PCL is only 8 bits so any offset that goes
;beyond 255 will cause problems because of a jump across a memory page.  This 
;code checks if the PCL+offset goes over 255 by checking the carry bit.  If the
;carry bit is set, 1 is added to the high byte of the program counter PCLATH to jump
;the next page.
;
;Put the offset value in temp and then call text_display.
;
;*****************************************************************************
text_display
	movlw	LOW table			;get low byte of table's location
	addwf	temp,w				;add the offset to the low byte
	movlw	HIGH table			;get the high byte of table's location
	btfsc	STATUS,C			;check to see if a page has been crossed
	addlw	1					;if so, add 1 to the high byte
	movwf	PCLATH				;put the adjusted high byte into counter
	movfw	temp				;put offset in the w register
	call	table_get			;jump into the table
	return						;from text_display

;*****************************************************************************
;
	


;*****************************************************************************
;
;Display 10-bit decimal number (from ADC)
;put high byte in h_byte
;put low byte in l_byte
;call display_DEC
;
;*****************************************************************************
display_DEC
	call	b2_BCD
	swapf	r1,w
	andlw	b'00001111'
	addlw	.48
	call	LCDOutput
	movfw	r1
	andlw	b'00001111'
	addlw	.48
	call	LCDOutput
	swapf	r2,w
	andlw	b'00001111'
	addlw	.48
	call	LCDOutput
	movfw	r2
	andlw	b'00001111'
	addlw	.48
	call	LCDOutput
	return
;end display_DEC
;*******************************************************************************

;*****************************************************************************
;
;LCD Output
;put the desired value into the w register and call LCDOutput
;
;*****************************************************************************
;*****************************************************************************
;
;LCD Output
;put the desired value into the w register and call LCDOutput
;
;*****************************************************************************

LCDOutput

	movwf	byte_to_send
	movlw 	.8
	movwf 	bitcounter			;set up to send 8 bits

;sending start bit
;this ideally would be a subroutine, but stack limitations
;call for imbedding this into the code
	bcf		PORTA,LCD			;send start bit
	movlw	.30
	movwf	count		
	goto    $+1			
	goto	$+1
bit
	decfsz	count, f
	goto	bit

;sending next 8-bits after start bit
nextbit
	rrf		byte_to_send,f		;sending lsb first
	bsf		PORTA,LCD			;assume 1 bit
	btfss	STATUS,C
	bcf		PORTA,LCD			;if not a 1 bit then clear
;bitdelay
;this ideally would be a subroutine, but stack limitations
;call for imbedding this into the code
	movlw	.30
	movwf	count
	goto    $+1
	goto	$+1
bit1
	decfsz	count, f
	goto	bit1
	decfsz	bitcounter,f
	goto	nextbit

	bsf		PORTA,LCD			;set to high for resting state

    call    delay5mS			;need some delay time to allow LCD to react and complete
	return
;end LCDOutput
;*******************************************************************************


;******************************************************************************
;
;Binary to BCD converter
;Put binary value to be converted into h_byte and l_byte
;then call b2_BCD.  Decimal converstion is returned in r0, r1, and r2
;with MSB in lower nibble of r0.
;
;******************************************************************************
b2_BCD
	bcf		STATUS,C
	movlw	.16
	movwf	countBCD
	clrf	r0
	clrf	r1
	clrf	r2
loop16
	rlf		l_byte,f
	rlf		h_byte,f
	rlf		r2,f
	rlf		r1,f
	rlf		r0,f
	
	decfsz	countBCD,f
	goto	adjDEC
	retlw	0

adjDEC
	movlw	r2
	movwf	FSR
	call	adjBCD
	
	movlw	r1
	movwf	FSR
	call	adjBCD

	movlw	r0
	movwf	FSR
	call	adjBCD

	goto	loop16

adjBCD
	movlw	3
	addwf	0,w
	movwf	temp
	btfsc	temp,3
	movwf	0
	movlw	30
	addwf	0,w
	movwf	temp
	btfsc	temp,7
	movwf	0
	retlw	0
;end Binary to BCD Converter
;******************************************************************************	



;******************************************************************************
;Delay Routines
;******************************************************************************
;more delay options than needed, this is standard delay code I use in other programs
wait5sec
		call	wait1sec
		call	wait1sec
		call	wait1sec
		call	wait1sec		;1 seconds plus the one to follow
		

wait1sec
		call	wait300mS
		call	wait300mS
		call	wait300mS
		goto	wait100mS
wait300mS
        call    delay50mS
wait250mS
        call    delay50mS
        call    delay50mS
        call    delay50mS
wait100mS
        call    delay50mS
wait50mS
        call    delay50mS
		return
;delay1ms, is a very accurate 1mS delay for a 4Mhz clock.
delay1mS
        movlw   .198
        movwf   count
        nop
        goto    $+1
        goto    $+1
dly1mS        
        goto    $+1
        decfsz  count, F
        goto    dly1mS
        return
;delay5mS uses delay1mS to get a very accurate 5 mS delay
delay5mS
        call    delay1mS
        call    delay1mS
        call    delay1mS
        call    delay1mS
        call    delay1mS
        movlw   .4
        movwf   count
tweek5mS
        decfsz  count, F
        goto    tweek5mS
        return
;delay50mS uses delay1mS to get a very accurate 50mS delay
delay50mS
        movlw   .50
        movwf   tempa
dly50mS
        call    delay1mS
        decfsz  tempa, F
        goto    dly50mS
        movlw   .14
        movwf   count
tweek50mS
        decfsz  count, F
        goto    tweek50mS
        return
;delay200mS uses delay1mS to get a very accurate 200mS delay.
delay200mS
        movlw   .200
        movwf   tempa
dly200mS
        call    delay1mS
        decfsz  tempa, F
        goto    dly200mS
        movlw   .64
        movwf   count
tweek200mS
        decfsz  count, F
        goto    tweek200mS
        return

;bitdelay

;		movlw	.30				;this number works if the user uses the calbrated
;								;value for the internal clock. This routine, including
;								;the goto and nop statements below allow the user to develop
;								;an anticipated delay of 100uS for the bits at 9600 Baud.
;								;This delay can be verified by using the stop watch function
;								;of MPLab Simulator 
;		movwf	count
;		goto    $+1				;these goto statements allow you tweek the time of the
;		goto	$+1				;delay.  Goto statements like this take 2 clock cycles
;		nop						;while the nop statement takes 1 clock cycle to complete
;
;
;bit
;		decfsz	count, f
;		goto	bit
;		return
;end delay rountines



table_get
	movfw 	temp		;put the table offset into w
	addwf	PCL,f		;add the offset to the program counter to jump to character
table	dt	LCD_LINE0,'P','o','t'
	


	end
